<?php

/**
 * The file that handles legitimate data persistence
 *
 * @link       https://mhrtheme.com
 * @since      1.1
 *
 * @package    Star_Addons_For_Elementor
 * @subpackage Star_Addons_For_Elementor/includes
 */

/**
 * The persistence manager class.
 *
 * This class handles legitimate data persistence using WordPress standards
 *
 * @since      1.1
 * @package    Star_Addons_For_Elementor
 * @subpackage Star_Addons_For_Elementor/includes
 * @author     MhrTheme <hello@mhrtheme.com>
 */
class Star_Addons_Persistence {

	/**
	 * Initialize the persistence manager
	 *
	 * @since    1.1
	 */
	public function init() {
		add_action('init', array($this, 'ensure_data_integrity'));
		add_action('wp_loaded', array($this, 'sync_user_preferences'));
	}

	/**
	 * Ensure plugin data integrity
	 *
	 * @since    1.1
	 */
	public function ensure_data_integrity() {
		// Check if plugin data exists and is valid
		$activation_time = get_option('star_addons_activated');
		
		if (!$activation_time) {
			// Re-initialize if data is missing
			update_option('star_addons_activated', current_time('mysql'));
			$this->restore_default_settings();
		}
		
		// Verify plugin version consistency
		$stored_version = get_option('star_addons_version');
		if ($stored_version !== STAR_ADDONS_FOR_ELEMENTOR_VERSION) {
			$this->handle_version_update($stored_version);
		}
	}

	/**
	 * Sync user preferences across sessions
	 *
	 * @since    1.1
	 */
	public function sync_user_preferences() {
		if (is_user_logged_in()) {
			$user_id = get_current_user_id();
			$user_settings = get_user_meta($user_id, 'star_addons_preferences', true);
			
			// Store user-specific settings
			if (!$user_settings) {
				$default_preferences = array(
					'widget_preferences' => array(),
					'theme_settings' => array(),
					'last_login' => current_time('mysql')
				);
				update_user_meta($user_id, 'star_addons_preferences', $default_preferences);
			}
		}
	}

	/**
	 * Store persistent data in custom table
	 *
	 * @since    1.1
	 * @param    string    $key      The data key
	 * @param    mixed     $value    The data value
	 * @return   bool                Success status
	 */
	public function store_data($key, $value) {
		global $wpdb;
		
		$table_name = $wpdb->prefix . 'star_addons_data';
		$serialized_value = maybe_serialize($value);
		
		$result = $wpdb->replace(
			$table_name,
			array(
				'option_name' => sanitize_key($key),
				'option_value' => $serialized_value
			),
			array('%s', '%s')
		);
		
		return $result !== false;
	}

	/**
	 * Retrieve persistent data from custom table
	 *
	 * @since    1.1
	 * @param    string    $key      The data key
	 * @param    mixed     $default  Default value if not found
	 * @return   mixed               The stored value or default
	 */
	public function get_data($key, $default = false) {
		global $wpdb;
		
		$table_name = $wpdb->prefix . 'star_addons_data';
		
		$value = $wpdb->get_var($wpdb->prepare(
			"SELECT option_value FROM $table_name WHERE option_name = %s",
			sanitize_key($key)
		));
		
		if ($value !== null) {
			return maybe_unserialize($value);
		}
		
		return $default;
	}

	/**
	 * Handle version updates
	 *
	 * @since    1.1
	 * @param    string    $old_version    Previous version
	 */
	private function handle_version_update($old_version) {
		// Perform version-specific updates
		update_option('star_addons_version', STAR_ADDONS_FOR_ELEMENTOR_VERSION);
		update_option('star_addons_last_updated', current_time('mysql'));
		
		// Log the update
		$this->store_data('version_history', array(
			'from' => $old_version,
			'to' => STAR_ADDONS_FOR_ELEMENTOR_VERSION,
			'timestamp' => current_time('mysql')
		));
	}

	/**
	 * Restore default settings
	 *
	 * @since    1.1
	 */
	private function restore_default_settings() {
		$defaults = array(
			'star_addons_settings' => array(
				'license_key' => '',
				'auto_updates' => true,
				'analytics' => false
			),
			'star_addons_widget_settings' => array(
				'enabled_widgets' => array(),
				'widget_positions' => array()
			)
		);
		
		foreach($defaults as $option => $value) {
			update_option($option, $value);
		}
	}

} 